package com.heckmansoft.surjey.model.manager;

import java.util.Iterator;

import org.apache.log4j.Logger;

import com.heckmansoft.surjey.model.datastore.DatastoreException;
import com.heckmansoft.surjey.model.datastore.SurveyDatastore;
import com.heckmansoft.surjey.model.datastore.Transaction;
import com.heckmansoft.surjey.model.vo.Section;
import com.heckmansoft.surjey.model.vo.Survey;
import com.heckmansoft.surjey.model.vo.lazy.LazySurvey;
import com.heckmansoft.surjey.model.vo.list.ValueObjectList;

/**
 * Manages Survey objects.
 */
public class SurveyManager extends SurveyDatastore
{
    SectionManager sectionManager;

    private Logger log = Logger.getLogger(SurveyManager.class);

    /**
     * Constructor.
     */
    public SurveyManager() {
        super();
        sectionManager = new SectionManager();
    }

    /**
     * Creates a managed Survey instance from survey.
     * Creates all subordinate sections as well.
     * @param survey A survey instance
     * @return true if managed Survey instance was created successfully
     */
    public boolean insert(Survey survey) throws DatastoreException {
        log.debug("create(" + survey + ")");

        Transaction transaction = getTransaction();

        // Create the survey
        boolean status = super.insert(survey);
        String surveyId = survey.getId();

        // Create all subordinate sections
        if (! survey.sectionsIsNull()) {
            ValueObjectList sections = survey.getSections();

            Iterator listIterator = sections.iterator();
            while(listIterator.hasNext()) {
                Section section = (Section) listIterator.next();
                section.setSurveyId(surveyId);
            }
            status &= sectionManager.insert(sections);
        }

        transaction.finish(status);

        return status;
    }

    /**
     * Updates a particular survey.
     * Updates all subordinate sections as well.
     * @param survey Survey instance containing updated data.
     * @return true if survey is found and updated.
     */
    public boolean update(Survey survey) throws DatastoreException {
        log.debug("update(" + survey + ")");

        Transaction transaction = getTransaction();

        // Update the survey
        boolean status = super.update(survey);
        String surveyId = survey.getId();

        // Update all subordinate sections
        if (! survey.sectionsIsNull()) {
            Iterator listIterator = survey.getSections().iterator();
            while(listIterator.hasNext()) {
                Section section = (Section) listIterator.next();
                section.setSurveyId(surveyId);
                status &= sectionManager.update(section);
            }
        }

        transaction.finish(status);

        return status;

    }

    /**
     * Removes given survey.
     *
     * @param survey Survey to be removed.
     * @return true if a matching survey is found and removed
     */
    public boolean remove(Survey survey) throws DatastoreException {
        return remove(survey.getId());
    }

    /**
     * Removes survey with given id.
     *
     * @param id Id of survey to be removed.
     * @return true if a matching survey is found and removed
     */
    public boolean remove(String id) throws DatastoreException {
        log.debug("remove(" + id + ")");

        Transaction transaction = getTransaction();

        // Remove the section
        boolean status = super.remove(id);

        // Remove all subordinate questions
        status &= sectionManager.removeBySurveyId(id);

        transaction.finish(status);

        return status;
    }

    /**
     * Get all sections for survey with given id
     * @param id Id of survey to find sections for.
     * @return List of sections
     */
    public ValueObjectList getSections(String id) throws DatastoreException {
        return sectionManager.findBySurveyId(id);
    }

    /**
     * Create a new survey.
     * @return newly created survey.
     */
    public Survey createSurvey() {
        return new LazySurvey(this);
    }

}
