package com.heckmansoft.surjey.model.manager;

import java.util.Iterator;
import org.apache.log4j.Logger;

import com.heckmansoft.surjey.model.datastore.DatastoreException;
import com.heckmansoft.surjey.model.datastore.ResponseDatastore;
import com.heckmansoft.surjey.model.datastore.ResponseAnswerDatastore;
import com.heckmansoft.surjey.model.datastore.Transaction;
import com.heckmansoft.surjey.model.vo.ResponseAnswer;
import com.heckmansoft.surjey.model.vo.Response;
import com.heckmansoft.surjey.model.vo.lazy.LazyResponse;
import com.heckmansoft.surjey.model.vo.list.ValueObjectList;

/**
 * Manages Response objects.
 */
public class ResponseManager extends ResponseDatastore {
    private ResponseAnswerDatastore answersDatastore;

    private Logger log = Logger.getLogger(ResponseManager.class);

    public ResponseManager() {
        super();
        this.answersDatastore = new ResponseAnswerDatastore();
    }

    /**
     * Creates a managed Response instance from response.
     *
     * @param response A response instance
     * @return true if managed Response instance was created successfully
     */
    public boolean insert(Response response) throws DatastoreException {
        log.debug("insert(" + response + ")");

        Transaction transaction = getTransaction();

        // Create the response
        boolean status = super.insert(response);
        String responseId = response.getId();

        // Create all subordinate answers
        if (! response.hasNoAnswers()) {
            ValueObjectList answers = response.getResponseAnswers();

            // Set the responseId for each answer
            ResponseAnswer answer;
            Iterator iterator = answers.iterator();
            while (iterator.hasNext()) {
                answer = ((ResponseAnswer) iterator.next());
                answer.setResponseId(responseId);
            }

            // Create list of answers
            status &= answersDatastore.insert(answers);
        }

        // Commit or rollback
        transaction.finish(status);

        return status;

    }

    /**
     * Removes given response.
     *
     * @param response Response to be removed.
     * @return true if a matching response is found and removed
     */
    public boolean remove(Response response) throws DatastoreException {
        return remove(response.getId());
    }

    /**
     * Removes response with given id.
     *
     * @param id Id of response to be removed.
     * @return true if a matching response is found and removed
     */
    public boolean remove(String id) throws DatastoreException {
        log.debug("remove(" + id + ")");

        Transaction transaction = getTransaction();

        boolean status = super.remove(id);

        status &= answersDatastore.removeByResponseId(id);

        transaction.finish(status);

        return status;
    }

    /**
     * For a given response id, get the subordinate answers.
     * @param id Response id to get the answers for.
     * @return List of answers in response
     */
    public ValueObjectList getResponseAnswers(String id) throws DatastoreException {
        return answersDatastore.findByResponseId(id);
    }

    /**
     * Create a new response.
     * @return newly created response.
     */
    public Response createResponse() {
        return new LazyResponse(this);
    }

}
