package com.heckmansoft.surjey.model.datastore;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.List;

import org.apache.log4j.Logger;

import com.heckmansoft.surjey.model.vo.Response;
import com.heckmansoft.surjey.model.vo.ValueObject;

/**
 * Manages object-relational mapping for Response objects.
 */
public class ResponseDatastore extends Datastore {
    final static String TABLE_NAME = "responses";
    final static String TABLE_ADD_FIELDS = "survey_id, respondent_address";

    private Logger log = Logger.getLogger(ResponseDatastore.class);

    /**
     * Constructor.
     */
    public ResponseDatastore() {
        super(TABLE_NAME,TABLE_ADD_FIELDS);
    }

    /**
     * Finds a particular response.
     * @param id Id of the response to find.
     * @return A Response instance, or null if no response found
     */
    public Response findById(String id) throws DatastoreException {
        log.debug("findById(" + id + ")");
        return ((Response) extractSingle(selectById(id)));
    }

    /**
     * Finds a particular response.
     * @param surveyId Survey id of the survey for which to find responses.
     * @param address Address of the respondent for which to find responses.
     * @return A List of Response instances
     */
    public List findBySurveyIdAndAddress(String surveyId, String address) throws DatastoreException {
        log.debug("findBySurveyIdAndAddress(" + surveyId + "," + address + ")");
        List args = new ArrayList();
        args.add(Integer.valueOf(surveyId));
        args.add(address);
        return extractList(selectWhere("survey_id = ? AND respondent_address = ?",args));
    }

    /**
     * From the given object, get a list of arguments to use for
     * constructing a query or update.
     * @param vo Object containing data to use.
     * @return A list of arguments.
     */
    protected List getArgs(ValueObject vo) {
        List args = super.getArgs(vo);
        Response response = (Response) vo;
        args.add(Integer.valueOf(response.getSurveyId()));
        args.add(response.getRespondentAddress());
        return args;
    }

    /**
     * Extract an object from the given result set.
     * @param results The result set.
     * @return The newly extracted object.
     */
    protected ValueObject extract(ResultSet results) throws DatastoreException {
        try {
            // Create a new response
            Response response = createResponse();
            super.extract(results, response);

            response.setSurveyId(results.getString("survey_id"));
            response.setRespondentAddress(results.getString("respondent_address"));

            return response;

        } catch (SQLException e) {
            log.debug(e);
            throw new RuntimeException(e);
        }
    }

    /**
     * Create a new response.
     * @return newly created response.
     */
    protected Response createResponse() {
        return new Response();
    }

}
