package com.heckmansoft.surjey.model.datastore;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.List;

import org.apache.log4j.Logger;

import com.heckmansoft.surjey.model.vo.ResponseAnswer;
import com.heckmansoft.surjey.model.vo.ValueObject;
import com.heckmansoft.surjey.model.vo.list.ValueObjectList;

/**
 * Manages object-relational mapping for ResponseAnswer objects.
 */
public class ResponseAnswerDatastore extends Datastore {
    final static String TABLE_NAME = "response_answers";
    final static String TABLE_ADD_FIELDS = "response_id, question_id, answer";
    final static String TABLE_PARENT_ID_FIELD = "response_id";

    String parentIdField;

    private Logger log = Logger.getLogger(ResponseDatastore.class);

    /**
     * Constructor.
     */
    public ResponseAnswerDatastore() {
        super(TABLE_NAME,TABLE_ADD_FIELDS);
        parentIdField = TABLE_PARENT_ID_FIELD;
    }

    /**
     * Finds a particular question.
     *
     * @param id the id of the question to find.
     * @return A Question instance, or null if no match found
     */
    public ResponseAnswer findById(String id) throws DatastoreException {
        log.debug("findById(" + id + ")");
        return ((ResponseAnswer) extractSingle(selectById(id)));
    }

    /**
     * Finds all answers for a particular response.
     *
     * @param id Id of the response containing the answers to find.
     * @return A list of matching Answer instances
     */
    public ValueObjectList findByResponseId(String id) throws DatastoreException {
        log.debug("findByResponseId(" + id + ")");
        List args = new ArrayList();
        args.add(Integer.valueOf(id));
        return extractList(selectWhere(parentIdField + " = ?", args));
    }

    /**
     * Removes all answers for given response id.
     *
     * @param id Id of answer to be removed.
     * @return true if a matching answer is found and removed
     */
    public boolean removeByResponseId(String id) throws DatastoreException {
        log.debug("removeByResponseId(" + id + ")");
        ValueObjectList target = findByResponseId(id);
        if (target.size() == 0) return true;
        return remove(target);
    }

    /**
     * From the given object, get a list of arguments to use for
     * constructing a query or update.
     * @param vo Object containing data to use.
     * @return A list of arguments.
     */
    protected List getArgs(ValueObject vo) {
        List args = super.getArgs(vo);
        ResponseAnswer answer = (ResponseAnswer) vo;
        args.add(Integer.valueOf(answer.getResponseId()));
        args.add(Integer.valueOf(answer.getQuestionId()));
        args.add(answer.getAnswer());
        return args;
    }

    /**
     * Extract an object from the given result set.
     * @param results The result set.
     * @return The newly extracted object.
     */
    protected ValueObject extract(ResultSet results) throws DatastoreException {
        try {
            // Create a new answer
            ResponseAnswer answer = createResponseAnswer();
            super.extract(results, answer);

            answer.setResponseId(results.getString("response_id"));
            answer.setQuestionId(results.getString("question_id"));
            answer.setAnswer(results.getString("answer"));

            return answer;

        } catch (SQLException e) {
            log.debug(e);
            throw new RuntimeException(e);
        }
    }

    /**
     * Create a new response.
     * @return newly created response.
     */
    protected ResponseAnswer createResponseAnswer() {
        return new ResponseAnswer();
    }

}
